/*
 * Decompiled with CFR 0.152.
 */
package gov.nasa.worldwind.terrain;

import com.sun.opengl.util.BufferUtil;
import gov.nasa.worldwind.Configuration;
import gov.nasa.worldwind.View;
import gov.nasa.worldwind.WWObjectImpl;
import gov.nasa.worldwind.WorldWind;
import gov.nasa.worldwind.cache.BasicMemoryCache;
import gov.nasa.worldwind.cache.MemoryCache;
import gov.nasa.worldwind.geom.Angle;
import gov.nasa.worldwind.geom.Cylinder;
import gov.nasa.worldwind.geom.Extent;
import gov.nasa.worldwind.geom.Frustum;
import gov.nasa.worldwind.geom.Intersection;
import gov.nasa.worldwind.geom.LatLon;
import gov.nasa.worldwind.geom.Line;
import gov.nasa.worldwind.geom.Plane;
import gov.nasa.worldwind.geom.Position;
import gov.nasa.worldwind.geom.Sector;
import gov.nasa.worldwind.geom.Triangle;
import gov.nasa.worldwind.geom.Vec4;
import gov.nasa.worldwind.globes.Globe;
import gov.nasa.worldwind.pick.PickSupport;
import gov.nasa.worldwind.pick.PickedObject;
import gov.nasa.worldwind.render.DrawContext;
import gov.nasa.worldwind.terrain.SectorGeometry;
import gov.nasa.worldwind.terrain.SectorGeometryList;
import gov.nasa.worldwind.terrain.Tessellator;
import gov.nasa.worldwind.util.Logging;
import java.awt.Color;
import java.awt.Point;
import java.nio.DoubleBuffer;
import java.nio.IntBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import javax.media.opengl.GL;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class RectangularTessellator
extends WWObjectImpl
implements Tessellator {
    private static final double DEFAULT_LOG10_RESOLUTION_TARGET = 1.3;
    private static final int DEFAULT_MAX_LEVEL = 17;
    private static final int DEFAULT_NUM_LAT_SUBDIVISIONS = 5;
    private static final int DEFAULT_NUM_LON_SUBDIVISIONS = 10;
    private static final int DEFAULT_DENSITY = 20;
    private static final String CACHE_NAME = "Terrain";
    private static final String CACHE_ID = RectangularTessellator.class.getName();
    private static final HashMap<Integer, DoubleBuffer> parameterizations = new HashMap();
    private static final HashMap<Integer, IntBuffer> indexLists = new HashMap();
    private ArrayList<RectTile> topLevels;
    private PickSupport pickSupport = new PickSupport();
    private SectorGeometryList currentTiles = new SectorGeometryList();
    private Frustum currentFrustum;
    private Sector currentCoverage;
    private boolean makeTileSkirts = true;
    private int currentLevel;
    private int maxLevel = 17;
    private Globe globe;
    private int density = 20;

    @Override
    public SectorGeometryList tessellate(DrawContext drawContext) {
        if (drawContext == null) {
            String string = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (drawContext.getView() == null) {
            String string = Logging.getMessage("nullValue.ViewIsNull");
            Logging.logger().severe(string);
            throw new IllegalStateException(string);
        }
        if (!WorldWind.getMemoryCacheSet().containsCache(CACHE_ID)) {
            long l = Configuration.getLongValue("gov.nasa.worldwind.avkey.SectorGeometryCacheSize", 20000000L);
            BasicMemoryCache basicMemoryCache = new BasicMemoryCache((long)(0.85 * (double)l), l);
            basicMemoryCache.setName(CACHE_NAME);
            WorldWind.getMemoryCacheSet().addCache(CACHE_ID, basicMemoryCache);
        }
        this.maxLevel = Configuration.getIntegerValue("gov.nasa.worldwind.avkey.RectangularTessellatorMaxLevel", 17);
        if (this.topLevels == null) {
            this.topLevels = this.createTopLevelTiles(drawContext);
        }
        this.currentTiles.clear();
        this.currentLevel = 0;
        this.currentCoverage = null;
        this.currentFrustum = drawContext.getView().getFrustumInModelCoordinates();
        for (SectorGeometry sectorGeometry : this.topLevels) {
            this.selectVisibleTiles(drawContext, (RectTile)sectorGeometry);
        }
        this.currentTiles.setSector(this.currentCoverage);
        for (SectorGeometry sectorGeometry : this.currentTiles) {
            this.makeVerts(drawContext, (RectTile)sectorGeometry);
        }
        return this.currentTiles;
    }

    private ArrayList<RectTile> createTopLevelTiles(DrawContext drawContext) {
        ArrayList<RectTile> arrayList = new ArrayList<RectTile>(50);
        this.globe = drawContext.getGlobe();
        double d = 36.0;
        double d2 = 36.0;
        Angle angle = Angle.NEG90;
        for (int i = 0; i < 5; ++i) {
            Angle angle2 = angle.addDegrees(d);
            if (angle2.getDegrees() + 1.0 > 90.0) {
                angle2 = Angle.POS90;
            }
            Angle angle3 = Angle.NEG180;
            for (int j = 0; j < 10; ++j) {
                Angle angle4 = angle3.addDegrees(d2);
                if (angle4.getDegrees() + 1.0 > 180.0) {
                    angle4 = Angle.POS180;
                }
                Sector sector = new Sector(angle, angle2, angle3, angle4);
                arrayList.add(this.createTile(drawContext, sector, 0));
                angle3 = angle4;
            }
            angle = angle2;
        }
        return arrayList;
    }

    private RectTile createTile(DrawContext drawContext, Sector sector, int n) {
        Cylinder cylinder = drawContext.getGlobe().computeBoundingCylinder(drawContext.getVerticalExaggeration(), sector);
        double d = sector.getDeltaLatRadians() * drawContext.getGlobe().getRadius() / (double)this.density;
        return new RectTile(this, cylinder, n, this.density, sector, d);
    }

    @Override
    public boolean isMakeTileSkirts() {
        return this.makeTileSkirts;
    }

    @Override
    public void setMakeTileSkirts(boolean bl) {
        this.makeTileSkirts = bl;
    }

    private void selectVisibleTiles(DrawContext drawContext, RectTile rectTile) {
        Extent extent = rectTile.getExtent();
        if (extent != null && !extent.intersects(this.currentFrustum)) {
            return;
        }
        if (this.currentLevel < this.maxLevel - 1 && this.needToSplit(drawContext, rectTile)) {
            RectTile[] rectTileArray;
            ++this.currentLevel;
            for (RectTile rectTile2 : rectTileArray = this.split(drawContext, rectTile)) {
                this.selectVisibleTiles(drawContext, rectTile2);
            }
            --this.currentLevel;
            return;
        }
        this.currentCoverage = rectTile.getSector().union(this.currentCoverage);
        this.currentTiles.add(rectTile);
    }

    private boolean needToSplit(DrawContext drawContext, RectTile rectTile) {
        Vec4[] vec4Array = rectTile.sector.computeCornerPoints(drawContext.getGlobe(), drawContext.getVerticalExaggeration());
        Vec4 vec4 = rectTile.sector.computeCenterPoint(drawContext.getGlobe(), drawContext.getVerticalExaggeration());
        View view = drawContext.getView();
        double d = view.getEyePoint().distanceTo3(vec4Array[0]);
        double d2 = view.getEyePoint().distanceTo3(vec4Array[1]);
        double d3 = view.getEyePoint().distanceTo3(vec4Array[2]);
        double d4 = view.getEyePoint().distanceTo3(vec4Array[3]);
        double d5 = view.getEyePoint().distanceTo3(vec4);
        double d6 = d;
        if (d2 < d6) {
            d6 = d2;
        }
        if (d3 < d6) {
            d6 = d3;
        }
        if (d4 < d6) {
            d6 = d4;
        }
        if (d5 < d6) {
            d6 = d5;
        }
        double d7 = Math.log10(d6);
        double d8 = this.computeTileResolutionTarget(drawContext, rectTile);
        boolean bl = rectTile.log10CellSize <= d7 - d8;
        return !bl;
    }

    private double computeTileResolutionTarget(DrawContext drawContext, RectTile rectTile) {
        double d = drawContext.getGlobe().getElevationModel().getDetailHint(rectTile.sector);
        return 1.3 + d;
    }

    private RectTile[] split(DrawContext drawContext, RectTile rectTile) {
        Sector[] sectorArray = rectTile.sector.subdivide();
        RectTile[] rectTileArray = new RectTile[]{this.createTile(drawContext, sectorArray[0], rectTile.level + 1), this.createTile(drawContext, sectorArray[1], rectTile.level + 1), this.createTile(drawContext, sectorArray[2], rectTile.level + 1), this.createTile(drawContext, sectorArray[3], rectTile.level + 1)};
        return rectTileArray;
    }

    private CacheKey createCacheKey(DrawContext drawContext, RectTile rectTile) {
        return new CacheKey(drawContext, rectTile.sector, rectTile.density);
    }

    private void makeVerts(DrawContext drawContext, RectTile rectTile) {
        MemoryCache memoryCache = WorldWind.getMemoryCache(CACHE_ID);
        CacheKey cacheKey = this.createCacheKey(drawContext, rectTile);
        rectTile.ri = (RenderInfo)memoryCache.getObject(cacheKey);
        if (rectTile.ri != null && rectTile.ri.time >= System.currentTimeMillis() - 1000L) {
            return;
        }
        rectTile.ri = this.buildVerts(drawContext, rectTile, this.makeTileSkirts);
        if (rectTile.ri != null) {
            cacheKey = this.createCacheKey(drawContext, rectTile);
            memoryCache.add(cacheKey, rectTile.ri, rectTile.ri.getSizeInBytes());
        }
    }

    public RenderInfo buildVerts(DrawContext drawContext, RectTile rectTile, boolean bl) {
        int n = rectTile.density;
        int n2 = (n + 3) * (n + 3);
        DoubleBuffer doubleBuffer = BufferUtil.newDoubleBuffer(n2 * 3);
        ArrayList<LatLon> arrayList = this.computeLocations(rectTile);
        double[] dArray = new double[arrayList.size()];
        drawContext.getGlobe().getElevations(rectTile.sector, arrayList, rectTile.getResolution(), dArray);
        int n3 = 0;
        double d = drawContext.getVerticalExaggeration();
        double d2 = bl ? this.globe.getMinElevation() * d : 0.0;
        LatLon latLon = rectTile.sector.getCentroid();
        Vec4 vec4 = this.globe.computePointFromPosition(latLon.getLatitude(), latLon.getLongitude(), 0.0);
        int n4 = 0;
        Iterator<LatLon> iterator = arrayList.iterator();
        for (int i = 0; i <= n + 2; ++i) {
            for (int j = 0; j <= n + 2; ++j) {
                LatLon latLon2 = iterator.next();
                double d3 = d * dArray[n4++];
                if (i == 0 || i >= rectTile.density + 2 || j == 0 || j >= rectTile.density + 2) {
                    d3 -= d2 >= 0.0 ? d2 : -d2;
                }
                Vec4 vec42 = this.globe.computePointFromPosition(latLon2.getLatitude(), latLon2.getLongitude(), d3);
                doubleBuffer.put(n3++, vec42.x - vec4.x).put(n3++, vec42.y - vec4.y).put(n3++, vec42.z - vec4.z);
            }
        }
        return new RenderInfo(n, doubleBuffer, RectangularTessellator.getTextureCoordinates(n), vec4);
    }

    private ArrayList<LatLon> computeLocations(RectTile rectTile) {
        int n = rectTile.density;
        int n2 = (n + 3) * (n + 3);
        Angle angle = rectTile.sector.getMaxLatitude();
        Angle angle2 = rectTile.sector.getDeltaLat().divide(n);
        Angle angle3 = rectTile.sector.getMinLatitude();
        Angle angle4 = rectTile.sector.getMinLongitude();
        Angle angle5 = rectTile.sector.getMaxLongitude();
        Angle angle6 = rectTile.sector.getDeltaLon().divide(n);
        ArrayList<LatLon> arrayList = new ArrayList<LatLon>(n2);
        for (int i = 0; i <= n + 2; ++i) {
            Angle angle7 = angle4;
            for (int j = 0; j <= n + 2; ++j) {
                arrayList.add(new LatLon(angle3, angle7));
                if (j > n) {
                    angle7 = angle5;
                } else if (j != 0) {
                    angle7 = angle7.add(angle6);
                }
                if (angle7.degrees < -180.0) {
                    angle7 = Angle.NEG180;
                    continue;
                }
                if (!(angle7.degrees > 180.0)) continue;
                angle7 = Angle.POS180;
            }
            if (i > n) {
                angle3 = angle;
                continue;
            }
            if (i == 0) continue;
            angle3 = angle3.add(angle2);
        }
        return arrayList;
    }

    private void renderMultiTexture(DrawContext drawContext, RectTile rectTile, int n) {
        if (drawContext == null) {
            String string = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (n < 1) {
            String string = Logging.getMessage("generic.NumTextureUnitsLessThanOne");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        this.render(drawContext, rectTile, n);
    }

    private void render(DrawContext drawContext, RectTile rectTile) {
        if (drawContext == null) {
            String string = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        this.render(drawContext, rectTile, 1);
    }

    private long render(DrawContext drawContext, RectTile rectTile, int n) {
        if (rectTile.ri == null) {
            String string = Logging.getMessage("nullValue.RenderInfoIsNull");
            Logging.logger().severe(string);
            throw new IllegalStateException(string);
        }
        drawContext.getView().pushReferenceCenter(drawContext, rectTile.ri.referenceCenter);
        GL gL = drawContext.getGL();
        gL.glPushClientAttrib(2);
        gL.glEnableClientState(32884);
        gL.glVertexPointer(3, 5130, 0, rectTile.ri.vertices.rewind());
        for (int i = 0; i < n; ++i) {
            gL.glClientActiveTexture(33984 + i);
            gL.glEnableClientState(32888);
            Object object = drawContext.getValue("gov.nasa.worldwind.avkey.TextureCoordinates");
            if (object != null && object instanceof DoubleBuffer) {
                gL.glTexCoordPointer(2, 5130, 0, ((DoubleBuffer)object).rewind());
                continue;
            }
            gL.glTexCoordPointer(2, 5130, 0, rectTile.ri.texCoords.rewind());
        }
        gL.glDrawElements(5, rectTile.ri.indices.limit(), 5125, rectTile.ri.indices.rewind());
        gL.glPopClientAttrib();
        drawContext.getView().popReferenceCenter(drawContext);
        return rectTile.ri.indices.limit() - 2;
    }

    private void renderWireframe(DrawContext drawContext, RectTile rectTile, boolean bl, boolean bl2) {
        if (drawContext == null) {
            String string = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (rectTile.ri == null) {
            String string = Logging.getMessage("nullValue.RenderInfoIsNull");
            Logging.logger().severe(string);
            throw new IllegalStateException(string);
        }
        drawContext.getView().pushReferenceCenter(drawContext, rectTile.ri.referenceCenter);
        GL gL = drawContext.getGL();
        gL.glPushAttrib(270601);
        gL.glEnable(3042);
        gL.glBlendFunc(770, 1);
        gL.glDisable(2929);
        gL.glEnable(2884);
        gL.glCullFace(1029);
        gL.glDisable(3553);
        gL.glColor4d(1.0, 1.0, 1.0, 0.2);
        gL.glPolygonMode(1028, 6913);
        if (bl) {
            gL.glPushClientAttrib(2);
            gL.glEnableClientState(32884);
            gL.glVertexPointer(3, 5130, 0, rectTile.ri.vertices.rewind());
            gL.glDrawElements(5, rectTile.ri.indices.limit(), 5125, rectTile.ri.indices.rewind());
            gL.glPopClientAttrib();
        }
        drawContext.getView().popReferenceCenter(drawContext);
        if (bl2) {
            this.renderPatchBoundary(drawContext, rectTile, gL);
        }
        gL.glPopAttrib();
    }

    private void renderPatchBoundary(DrawContext drawContext, RectTile rectTile, GL gL) {
        gL.glColor4d(1.0, 0.0, 0.0, 1.0);
        Vec4[] vec4Array = rectTile.sector.computeCornerPoints(drawContext.getGlobe(), drawContext.getVerticalExaggeration());
        gL.glBegin(7);
        gL.glVertex3d(vec4Array[0].x, vec4Array[0].y, vec4Array[0].z);
        gL.glVertex3d(vec4Array[1].x, vec4Array[1].y, vec4Array[1].z);
        gL.glVertex3d(vec4Array[2].x, vec4Array[2].y, vec4Array[2].z);
        gL.glVertex3d(vec4Array[3].x, vec4Array[3].y, vec4Array[3].z);
        gL.glEnd();
    }

    private void renderBoundingVolume(DrawContext drawContext, RectTile rectTile) {
        Extent extent = rectTile.getExtent();
        if (extent == null) {
            return;
        }
        if (extent instanceof Cylinder) {
            ((Cylinder)extent).render(drawContext);
        }
    }

    private PickedObject[] pick(DrawContext drawContext, RectTile rectTile, List<? extends Point> list) {
        if (drawContext == null) {
            String string = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (list.size() == 0) {
            return null;
        }
        if (rectTile.ri == null) {
            return null;
        }
        PickedObject[] pickedObjectArray = new PickedObject[list.size()];
        this.renderTrianglesWithUniqueColors(drawContext, rectTile);
        for (int i = 0; i < list.size(); ++i) {
            pickedObjectArray[i] = this.resolvePick(drawContext, rectTile, list.get(i));
        }
        return pickedObjectArray;
    }

    private void pick(DrawContext drawContext, RectTile rectTile, Point point) {
        if (drawContext == null) {
            String string = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (rectTile.ri == null) {
            return;
        }
        this.renderTrianglesWithUniqueColors(drawContext, rectTile);
        PickedObject pickedObject = this.resolvePick(drawContext, rectTile, point);
        if (pickedObject != null) {
            drawContext.addPickedObject(pickedObject);
        }
    }

    private void renderTrianglesWithUniqueColors(DrawContext drawContext, RectTile rectTile) {
        if (drawContext == null) {
            String string = Logging.getMessage("nullValue.DrawContextIsNull");
            Logging.logger().severe(string);
            throw new IllegalStateException(string);
        }
        if (rectTile.ri.vertices == null) {
            return;
        }
        rectTile.ri.vertices.rewind();
        rectTile.ri.indices.rewind();
        GL gL = drawContext.getGL();
        if (null != rectTile.ri.referenceCenter) {
            drawContext.getView().pushReferenceCenter(drawContext, rectTile.ri.referenceCenter);
        }
        rectTile.minColorCode = drawContext.getUniquePickColor().getRGB();
        int n = rectTile.ri.indices.capacity() - 2;
        int[] nArray = new int[3];
        double[] dArray = new double[3];
        gL.glBegin(4);
        for (int i = 0; i < n; ++i) {
            Color color = drawContext.getUniquePickColor();
            gL.glColor3ub((byte)(color.getRed() & 0xFF), (byte)(color.getGreen() & 0xFF), (byte)(color.getBlue() & 0xFF));
            rectTile.ri.indices.position(i);
            rectTile.ri.indices.get(nArray);
            rectTile.ri.vertices.position(3 * nArray[0]);
            rectTile.ri.vertices.get(dArray);
            gL.glVertex3d(dArray[0], dArray[1], dArray[2]);
            rectTile.ri.vertices.position(3 * nArray[1]);
            rectTile.ri.vertices.get(dArray);
            gL.glVertex3d(dArray[0], dArray[1], dArray[2]);
            rectTile.ri.vertices.position(3 * nArray[2]);
            rectTile.ri.vertices.get(dArray);
            gL.glVertex3d(dArray[0], dArray[1], dArray[2]);
        }
        gL.glEnd();
        rectTile.maxColorCode = drawContext.getUniquePickColor().getRGB();
        if (null != rectTile.ri.referenceCenter) {
            drawContext.getView().popReferenceCenter(drawContext);
        }
    }

    private PickedObject resolvePick(DrawContext drawContext, RectTile rectTile, Point point) {
        int n = this.pickSupport.getTopColor(drawContext, point);
        if (n < rectTile.minColorCode || n > rectTile.maxColorCode) {
            return null;
        }
        double d = 1.0E-5f;
        int n2 = n - rectTile.minColorCode - 1;
        if (rectTile.ri.indices == null || n2 >= rectTile.ri.indices.capacity() - 2) {
            return null;
        }
        double d2 = ((RenderInfo)((RectTile)rectTile).ri).referenceCenter.x;
        double d3 = ((RenderInfo)((RectTile)rectTile).ri).referenceCenter.y;
        double d4 = ((RenderInfo)((RectTile)rectTile).ri).referenceCenter.z;
        int[] nArray = new int[3];
        rectTile.ri.indices.position(n2);
        rectTile.ri.indices.get(nArray);
        double[] dArray = new double[3];
        rectTile.ri.vertices.position(3 * nArray[0]);
        rectTile.ri.vertices.get(dArray);
        Vec4 vec4 = new Vec4(dArray[0] + d2, dArray[1] + d3, dArray[2] + d4);
        rectTile.ri.vertices.position(3 * nArray[1]);
        rectTile.ri.vertices.get(dArray);
        Vec4 vec42 = new Vec4(dArray[0] + d2, dArray[1] + d3, dArray[2] + d4);
        rectTile.ri.vertices.position(3 * nArray[2]);
        rectTile.ri.vertices.get(dArray);
        Vec4 vec43 = new Vec4(dArray[0] + d2, dArray[1] + d3, dArray[2] + d4);
        Vec4 vec44 = vec42.subtract3(vec4);
        Vec4 vec45 = vec43.subtract3(vec4);
        Vec4 vec46 = vec44.cross3(vec45);
        Line line = drawContext.getView().computeRayFromScreenPoint(point.getX(), point.getY());
        Vec4 vec47 = line.getOrigin().subtract3(vec4);
        double d5 = -vec46.dot3(vec47);
        double d6 = vec46.dot3(line.getDirection());
        if (Math.abs(d6) < d) {
            return null;
        }
        double d7 = d5 / d6;
        Vec4 vec48 = line.getOrigin().add3(line.getDirection().multiply3(d7));
        Position position = drawContext.getGlobe().computePositionFromPoint(vec48);
        double d8 = drawContext.getGlobe().getElevation(position.getLatitude(), position.getLongitude());
        Position position2 = new Position(position.getLatitude(), position.getLongitude(), d8 *= drawContext.getVerticalExaggeration());
        return new PickedObject(point, n, position2, position.getLatitude(), position.getLongitude(), d8, true);
    }

    private Intersection[] intersect(RectTile rectTile, Line line) {
        int n;
        if (line == null) {
            String string = Logging.getMessage("nullValue.LineIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (rectTile.ri.vertices == null) {
            return null;
        }
        Vec4 vec4 = line.getDirection().cross3(this.globe.computeSurfaceNormalAtPoint(line.getOrigin()));
        Plane plane = new Plane(vec4.x(), vec4.y(), vec4.z(), -line.getOrigin().dot3(vec4));
        if (!rectTile.getExtent().intersects(plane)) {
            return null;
        }
        Vec4 vec42 = line.getDirection().cross3(vec4);
        Plane plane2 = new Plane(vec42.x(), vec42.y(), vec42.z(), -line.getOrigin().dot3(vec42));
        if (!rectTile.getExtent().intersects(plane2)) {
            return null;
        }
        ArrayList<Intersection> arrayList = new ArrayList<Intersection>();
        int[] nArray = new int[rectTile.ri.indices.limit()];
        double[] dArray = new double[rectTile.ri.vertices.limit()];
        rectTile.ri.indices.rewind();
        rectTile.ri.vertices.rewind();
        rectTile.ri.indices.get(nArray, 0, nArray.length);
        rectTile.ri.vertices.get(dArray, 0, dArray.length);
        rectTile.ri.indices.rewind();
        rectTile.ri.vertices.rewind();
        int n2 = rectTile.ri.indices.capacity() - 2;
        double d = ((RenderInfo)((RectTile)rectTile).ri).referenceCenter.x;
        double d2 = ((RenderInfo)((RectTile)rectTile).ri).referenceCenter.y;
        double d3 = ((RenderInfo)((RectTile)rectTile).ri).referenceCenter.z;
        double d4 = rectTile.getSector().getDeltaLatRadians() * this.globe.getRadius() / (double)this.density;
        double d5 = Math.sqrt(d4 * d4 * 2.0) / 2.0;
        double d6 = ((Cylinder)rectTile.getExtent()).getCylinderHeight();
        int n3 = (this.density + 2) * 2 + 6;
        int n4 = n2 - n3;
        int n5 = -1;
        for (n = n3; n < n4; n += 2) {
            Vec4 vec43;
            Vec4 vec44;
            int n6 = n5 = n5 == this.density - 1 ? -4 : n5 + 1;
            if (n5 < 0) continue;
            int n7 = 3 * nArray[n + 1];
            Vec4 vec45 = new Vec4(dArray[n7++] + d, dArray[n7++] + d2, dArray[n7] + d3);
            n7 = 3 * nArray[n + 2];
            if (Math.abs(plane.distanceTo(vec44 = Vec4.mix3(0.5, vec45, vec43 = new Vec4(dArray[n7++] + d, dArray[n7++] + d2, dArray[n7] + d3)))) > d5 || Math.abs(plane2.distanceTo(vec44)) > d6) continue;
            n7 = 3 * nArray[n];
            Vec4 vec46 = new Vec4(dArray[n7++] + d, dArray[n7++] + d2, dArray[n7] + d3);
            n7 = 3 * nArray[n + 3];
            Vec4 vec47 = new Vec4(dArray[n7++] + d, dArray[n7++] + d2, dArray[n7] + d3);
            Triangle triangle = new Triangle(vec46, vec45, vec43);
            Vec4 vec48 = triangle.intersect(line);
            if (vec48 != null) {
                arrayList.add(new Intersection(vec48, false));
            }
            if ((vec48 = (triangle = new Triangle(vec45, vec43, vec47)).intersect(line)) == null) continue;
            arrayList.add(new Intersection(vec48, false));
        }
        n = arrayList.size();
        if (n == 0) {
            return null;
        }
        Intersection[] intersectionArray = new Intersection[n];
        arrayList.toArray(intersectionArray);
        final Vec4 vec49 = line.getOrigin();
        Arrays.sort(intersectionArray, new Comparator<Intersection>(){

            @Override
            public int compare(Intersection intersection, Intersection intersection2) {
                if (intersection == null && intersection2 == null) {
                    return 0;
                }
                if (intersection2 == null) {
                    return -1;
                }
                if (intersection == null) {
                    return 1;
                }
                Vec4 vec4 = intersection.getIntersectionPoint();
                Vec4 vec42 = intersection2.getIntersectionPoint();
                double d = vec49.distanceTo3(vec4);
                double d2 = vec49.distanceTo3(vec42);
                return Double.compare(d, d2);
            }
        });
        return intersectionArray;
    }

    private Intersection[] intersect(RectTile rectTile, double d) {
        int n;
        if (rectTile.ri.vertices == null) {
            return null;
        }
        Cylinder cylinder = (Cylinder)rectTile.getExtent();
        if (!(this.globe.isPointAboveElevation(cylinder.getBottomCenter(), d) ^ this.globe.isPointAboveElevation(cylinder.getTopCenter(), d))) {
            return null;
        }
        ArrayList<Intersection> arrayList = new ArrayList<Intersection>();
        int[] nArray = new int[rectTile.ri.indices.limit()];
        double[] dArray = new double[rectTile.ri.vertices.limit()];
        rectTile.ri.indices.rewind();
        rectTile.ri.vertices.rewind();
        rectTile.ri.indices.get(nArray, 0, nArray.length);
        rectTile.ri.vertices.get(dArray, 0, dArray.length);
        rectTile.ri.indices.rewind();
        rectTile.ri.vertices.rewind();
        int n2 = rectTile.ri.indices.capacity() - 2;
        double d2 = ((RenderInfo)((RectTile)rectTile).ri).referenceCenter.x;
        double d3 = ((RenderInfo)((RectTile)rectTile).ri).referenceCenter.y;
        double d4 = ((RenderInfo)((RectTile)rectTile).ri).referenceCenter.z;
        int n3 = (this.density + 2) * 2 + 6;
        int n4 = n2 - n3;
        int n5 = -1;
        for (n = n3; n < n4; n += 2) {
            int n6 = n5 = n5 == this.density - 1 ? -4 : n5 + 1;
            if (n5 < 0) continue;
            int n7 = 3 * nArray[n];
            Vec4 vec4 = new Vec4(dArray[n7++] + d2, dArray[n7++] + d3, dArray[n7] + d4);
            n7 = 3 * nArray[n + 1];
            Vec4 vec42 = new Vec4(dArray[n7++] + d2, dArray[n7++] + d3, dArray[n7] + d4);
            n7 = 3 * nArray[n + 2];
            Vec4 vec43 = new Vec4(dArray[n7++] + d2, dArray[n7++] + d3, dArray[n7] + d4);
            n7 = 3 * nArray[n + 3];
            Vec4 vec44 = new Vec4(dArray[n7++] + d2, dArray[n7++] + d3, dArray[n7] + d4);
            Intersection[] intersectionArray = this.globe.intersect(new Triangle(vec4, vec42, vec43), d);
            if (intersectionArray != null) {
                arrayList.add(intersectionArray[0]);
                arrayList.add(intersectionArray[1]);
            }
            if ((intersectionArray = this.globe.intersect(new Triangle(vec42, vec43, vec44), d)) == null) continue;
            arrayList.add(intersectionArray[0]);
            arrayList.add(intersectionArray[1]);
        }
        n = arrayList.size();
        if (n == 0) {
            return null;
        }
        Intersection[] intersectionArray = new Intersection[n];
        arrayList.toArray(intersectionArray);
        return intersectionArray;
    }

    private Vec4 getSurfacePoint(RectTile rectTile, Angle angle, Angle angle2, double d) {
        Vec4 vec4 = this.getSurfacePoint(rectTile, angle, angle2);
        if (d != 0.0 && vec4 != null) {
            vec4 = RectangularTessellator.applyOffset(this.globe, vec4, d);
        }
        return vec4;
    }

    private static Vec4 applyOffset(Globe globe, Vec4 vec4, double d) {
        Vec4 vec42 = globe.computeSurfaceNormalAtPoint(vec4);
        vec4 = Vec4.fromLine3(vec4, d, vec42);
        return vec4;
    }

    private Vec4 getSurfacePoint(RectTile rectTile, Angle angle, Angle angle2) {
        if (angle == null || angle2 == null) {
            String string = Logging.getMessage("nullValue.LatLonIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (!rectTile.sector.contains(angle, angle2)) {
            return null;
        }
        if (rectTile.ri == null) {
            return null;
        }
        double d = angle.getDegrees();
        double d2 = angle2.getDegrees();
        double d3 = rectTile.sector.getMinLatitude().getDegrees();
        double d4 = rectTile.sector.getMaxLatitude().getDegrees();
        double d5 = rectTile.sector.getMinLongitude().getDegrees();
        double d6 = rectTile.sector.getMaxLongitude().getDegrees();
        double d7 = (d2 - d5) / (d6 - d5);
        double d8 = (d - d3) / (d4 - d3);
        int n = (int)(d8 * (double)rectTile.density);
        int n2 = (int)(d7 * (double)rectTile.density);
        double d9 = RectangularTessellator.createPosition(n2, d7, rectTile.ri.density);
        double d10 = RectangularTessellator.createPosition(n, d8, rectTile.ri.density);
        Vec4 vec4 = RectangularTessellator.interpolate(n, n2, d9, d10, rectTile.ri);
        vec4 = vec4.add3(rectTile.ri.referenceCenter);
        return vec4;
    }

    private static double createPosition(int n, double d, int n2) {
        double d2 = (double)n / (double)n2;
        double d3 = (double)(n + 1) / (double)n2;
        return (d - d2) / (d3 - d2);
    }

    private static Vec4 interpolate(int n, int n2, double d, double d2, RenderInfo renderInfo) {
        int n3 = renderInfo.density + 3;
        int n4 = ++n * n3 + ++n2;
        int n5 = n3 * 3;
        double[] dArray = new double[6];
        renderInfo.vertices.position(n4 *= 3);
        renderInfo.vertices.get(dArray);
        Vec4 vec4 = new Vec4(dArray[0], dArray[1], dArray[2]);
        Vec4 vec42 = new Vec4(dArray[3], dArray[4], dArray[5]);
        renderInfo.vertices.position(n4 += n5);
        renderInfo.vertices.get(dArray);
        Vec4 vec43 = new Vec4(dArray[0], dArray[1], dArray[2]);
        Vec4 vec44 = new Vec4(dArray[3], dArray[4], dArray[5]);
        return RectangularTessellator.interpolate(vec4, vec42, vec44, vec43, d, d2);
    }

    private static Vec4 interpolate(Vec4 vec4, Vec4 vec42, Vec4 vec43, Vec4 vec44, double d, double d2) {
        double d3 = d + d2;
        if (d3 == 1.0) {
            return new Vec4(vec44.x * d2 + vec42.x * d, vec44.y * d2 + vec42.y * d, vec44.z * d2 + vec42.z * d);
        }
        if (d3 > 1.0) {
            Vec4 vec45 = vec44.subtract3(vec43).multiply3(1.0 - d);
            Vec4 vec46 = vec42.subtract3(vec43).multiply3(1.0 - d2);
            return vec43.add3(vec45).add3(vec46);
        }
        Vec4 vec47 = vec42.subtract3(vec4).multiply3(d);
        Vec4 vec48 = vec44.subtract3(vec4).multiply3(d2);
        return vec4.add3(vec47).add3(vec48);
    }

    private static double[] baryCentricCoordsRequireInside(Vec4 vec4, Vec4[] vec4Array) {
        double[] dArray = new double[3];
        double d = RectangularTessellator.distanceFromLine(vec4Array[0], vec4Array[1], vec4Array[2].subtract3(vec4Array[1]));
        double d2 = RectangularTessellator.distanceFromLine(vec4, vec4Array[1], vec4Array[2].subtract3(vec4Array[1]));
        dArray[0] = d2 / d;
        if (Math.abs(dArray[0]) < 1.0E-4) {
            dArray[0] = 0.0;
        } else if (Math.abs(1.0 - dArray[0]) < 1.0E-4) {
            dArray[0] = 1.0;
        }
        if (dArray[0] < 0.0 || dArray[0] > 1.0) {
            return null;
        }
        d = RectangularTessellator.distanceFromLine(vec4Array[1], vec4Array[0], vec4Array[2].subtract3(vec4Array[0]));
        d2 = RectangularTessellator.distanceFromLine(vec4, vec4Array[0], vec4Array[2].subtract3(vec4Array[0]));
        dArray[1] = d2 / d;
        if (Math.abs(dArray[1]) < 1.0E-4) {
            dArray[1] = 0.0;
        } else if (Math.abs(1.0 - dArray[1]) < 1.0E-4) {
            dArray[1] = 1.0;
        }
        if (dArray[1] < 0.0 || dArray[1] > 1.0) {
            return null;
        }
        dArray[2] = 1.0 - dArray[0] - dArray[1];
        if (Math.abs(dArray[2]) < 1.0E-4) {
            dArray[2] = 0.0;
        } else if (Math.abs(1.0 - dArray[2]) < 1.0E-4) {
            dArray[2] = 1.0;
        }
        if (dArray[2] < 0.0) {
            return null;
        }
        return dArray;
    }

    private static double distanceFromLine(Vec4 vec4, Vec4 vec42, Vec4 vec43) {
        double d;
        Vec4 vec44 = vec4.subtract3(vec42);
        double d2 = vec44.dot3(vec44);
        double d3 = vec43.normalize3().dot3(vec44);
        if ((d = d2 - (d3 *= d3)) < 0.0) {
            return 0.0;
        }
        return Math.sqrt(d);
    }

    protected DoubleBuffer makeGeographicTexCoords(SectorGeometry sectorGeometry, SectorGeometry.GeographicTextureCoordinateComputer geographicTextureCoordinateComputer) {
        int n;
        double[] dArray;
        Angle angle;
        int n2;
        if (sectorGeometry == null) {
            String string = Logging.getMessage("nullValue.SectorGeometryIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        if (geographicTextureCoordinateComputer == null) {
            String string = Logging.getMessage("nullValue.TextureCoordinateComputerIsNull");
            Logging.logger().severe(string);
            throw new IllegalArgumentException(string);
        }
        RectTile rectTile = (RectTile)sectorGeometry;
        int n3 = rectTile.density;
        if (n3 < 1) {
            n3 = 1;
        }
        int n4 = (n3 + 3) * (n3 + 3);
        DoubleBuffer doubleBuffer = BufferUtil.newDoubleBuffer(2 * n4);
        double d = rectTile.sector.getDeltaLatRadians() / (double)n3;
        double d2 = rectTile.sector.getDeltaLonRadians() / (double)n3;
        Angle angle2 = rectTile.sector.getMinLatitude();
        Angle angle3 = rectTile.sector.getMaxLatitude();
        Angle angle4 = rectTile.sector.getMinLongitude();
        Angle angle5 = rectTile.sector.getMaxLongitude();
        int n5 = 2 * (n3 + 3);
        for (n2 = 0; n2 < n3; ++n2) {
            angle = Angle.fromRadians(angle2.radians + (double)n2 * d);
            dArray = geographicTextureCoordinateComputer.compute(angle, angle4);
            doubleBuffer.put(n5++, dArray[0]).put(n5++, dArray[1]);
            for (int i = 0; i < n3; ++i) {
                Angle angle6 = Angle.fromRadians(angle4.radians + (double)i * d2);
                dArray = geographicTextureCoordinateComputer.compute(angle, angle6);
                doubleBuffer.put(n5++, dArray[0]).put(n5++, dArray[1]);
            }
            dArray = geographicTextureCoordinateComputer.compute(angle, angle5);
            doubleBuffer.put(n5++, dArray[0]).put(n5++, dArray[1]);
            doubleBuffer.put(n5++, dArray[0]).put(n5++, dArray[1]);
        }
        dArray = geographicTextureCoordinateComputer.compute(angle3, angle4);
        doubleBuffer.put(n5++, dArray[0]).put(n5++, dArray[1]);
        for (n2 = 0; n2 < n3; ++n2) {
            angle = Angle.fromRadians(angle4.radians + (double)n2 * d2);
            dArray = geographicTextureCoordinateComputer.compute(angle3, angle);
            doubleBuffer.put(n5++, dArray[0]).put(n5++, dArray[1]);
        }
        dArray = geographicTextureCoordinateComputer.compute(angle3, angle5);
        doubleBuffer.put(n5++, dArray[0]).put(n5++, dArray[1]);
        doubleBuffer.put(n5++, dArray[0]).put(n5++, dArray[1]);
        n2 = n5 - 2 * (n3 + 3);
        for (n = 0; n < n3 + 3; ++n) {
            doubleBuffer.put(n5++, doubleBuffer.get(n2++));
            doubleBuffer.put(n5++, doubleBuffer.get(n2++));
        }
        n5 = 0;
        n2 = 2 * (n3 + 3);
        for (n = 0; n < n3 + 3; ++n) {
            doubleBuffer.put(n5++, doubleBuffer.get(n2++));
            doubleBuffer.put(n5++, doubleBuffer.get(n2++));
        }
        return doubleBuffer;
    }

    private static DoubleBuffer getTextureCoordinates(int n) {
        int n2;
        int n3;
        DoubleBuffer doubleBuffer;
        if (n < 1) {
            n = 1;
        }
        if ((doubleBuffer = parameterizations.get(n)) != null) {
            return doubleBuffer;
        }
        int n4 = (n + 3) * (n + 3);
        doubleBuffer = BufferUtil.newDoubleBuffer(2 * n4);
        double d = 1.0 / (double)n;
        int n5 = 2 * (n + 3);
        for (int i = 0; i < n; ++i) {
            double d2 = (double)i * d;
            doubleBuffer.put(n5++, 0.0);
            doubleBuffer.put(n5++, d2);
            for (n3 = 0; n3 < n; ++n3) {
                doubleBuffer.put(n5++, (double)n3 * d);
                doubleBuffer.put(n5++, d2);
            }
            doubleBuffer.put(n5++, 0.999999);
            doubleBuffer.put(n5++, d2);
            doubleBuffer.put(n5++, 0.999999);
            doubleBuffer.put(n5++, d2);
        }
        double d3 = 0.999999;
        doubleBuffer.put(n5++, 0.0);
        doubleBuffer.put(n5++, d3);
        for (n2 = 0; n2 < n; ++n2) {
            doubleBuffer.put(n5++, (double)n2 * d);
            doubleBuffer.put(n5++, d3);
        }
        doubleBuffer.put(n5++, 0.999999);
        doubleBuffer.put(n5++, d3);
        doubleBuffer.put(n5++, 0.999999);
        doubleBuffer.put(n5++, d3);
        n2 = n5 - 2 * (n + 3);
        for (n3 = 0; n3 < n + 3; ++n3) {
            doubleBuffer.put(n5++, doubleBuffer.get(n2++));
            doubleBuffer.put(n5++, doubleBuffer.get(n2++));
        }
        n5 = 0;
        n2 = 2 * (n + 3);
        for (n3 = 0; n3 < n + 3; ++n3) {
            doubleBuffer.put(n5++, doubleBuffer.get(n2++));
            doubleBuffer.put(n5++, doubleBuffer.get(n2++));
        }
        parameterizations.put(n, doubleBuffer);
        return doubleBuffer;
    }

    protected static IntBuffer getIndices(int n) {
        IntBuffer intBuffer;
        if (n < 1) {
            n = 1;
        }
        if ((intBuffer = indexLists.get(n)) != null) {
            return intBuffer;
        }
        int n2 = n + 2;
        int n3 = 2 * n2 * n2 + 4 * n2 - 2;
        intBuffer = BufferUtil.newIntBuffer(n3);
        int n4 = 0;
        for (int i = 0; i < n2; ++i) {
            int n5;
            intBuffer.put(n4);
            if (i > 0) {
                intBuffer.put(++n4);
                intBuffer.put(n4);
            }
            if (i % 2 == 0) {
                intBuffer.put(++n4);
                for (n5 = 0; n5 < n2; ++n5) {
                    intBuffer.put(n4 += n2);
                    intBuffer.put(++n4);
                }
                continue;
            }
            intBuffer.put(--n4);
            for (n5 = 0; n5 < n2; ++n5) {
                intBuffer.put(n4 -= n2);
                intBuffer.put(--n4);
            }
        }
        indexLists.put(n, intBuffer);
        return intBuffer;
    }

    private SectorGeometry.ExtractedShapeDescription getIntersectingTessellationPieces(RectTile rectTile, Plane[] planeArray) {
        rectTile.ri.vertices.rewind();
        rectTile.ri.indices.rewind();
        Vec4 vec4 = rectTile.ri.referenceCenter;
        if (vec4 == null) {
            vec4 = new Vec4(0.0);
        }
        int n = rectTile.ri.indices.capacity() - 2;
        int[] nArray = new int[3];
        double[] dArray = new double[3];
        SectorGeometry.ExtractedShapeDescription extractedShapeDescription = null;
        for (int i = 0; i < n; ++i) {
            rectTile.ri.indices.position(i);
            rectTile.ri.indices.get(nArray);
            if (nArray[0] == nArray[1] || nArray[0] == nArray[2] || nArray[1] == nArray[2]) continue;
            Vec4[] vec4Array = new Vec4[3];
            for (int j = 0; j < 3; ++j) {
                rectTile.ri.vertices.position(3 * nArray[j]);
                rectTile.ri.vertices.get(dArray);
                vec4Array[j] = new Vec4(dArray[0] + vec4.getX(), dArray[1] + vec4.getY(), dArray[2] + vec4.getZ(), 1.0);
            }
            extractedShapeDescription = this.addClippedPolygon(vec4Array, planeArray, extractedShapeDescription);
        }
        return extractedShapeDescription;
    }

    private SectorGeometry.ExtractedShapeDescription addClippedPolygon(Vec4[] vec4Array, Plane[] planeArray, SectorGeometry.ExtractedShapeDescription extractedShapeDescription) {
        if (this.isSkirt(vec4Array)) {
            return extractedShapeDescription;
        }
        Vec4[] vec4Array2 = new Vec4[3];
        System.arraycopy(vec4Array, 0, vec4Array2, 0, 3);
        for (Plane plane : planeArray) {
            if ((vec4Array2 = this.doSHPass(plane, vec4Array2)) != null) continue;
            return extractedShapeDescription;
        }
        if (extractedShapeDescription == null) {
            extractedShapeDescription = new SectorGeometry.ExtractedShapeDescription(new ArrayList<Vec4[]>(), new ArrayList<SectorGeometry.BoundaryEdge>());
        }
        extractedShapeDescription.interiorPolys.add(vec4Array2);
        this.addBoundaryEdges(vec4Array2, vec4Array, extractedShapeDescription.shapeOutline);
        return extractedShapeDescription;
    }

    private boolean isSkirt(Vec4[] vec4Array) {
        Vec4 vec4;
        Vec4 vec42 = this.globe.computeSurfaceNormalAtPoint(vec4Array[0]);
        double d = Math.max(Math.abs(vec4Array[0].x), Math.abs(vec4Array[0].y));
        d = Math.max(d, Math.abs(vec4Array[0].z));
        Vec4 vec43 = vec4Array[0].divide3(d);
        Vec4 vec44 = vec4Array[1].divide3(d).subtract3(vec43);
        Vec4 vec45 = vec44.cross3(vec4 = vec4Array[vec4Array.length - 1].divide3(d).subtract3(vec43)).normalize3();
        return Math.abs(vec45.dot3(vec42)) < 1.0E-4;
    }

    private Vec4[] doSHPass(Plane plane, Vec4[] vec4Array) {
        ArrayList<Vec4> arrayList = new ArrayList<Vec4>();
        Vec4 vec4 = vec4Array[0];
        boolean bl = plane.dot(vec4) <= 0.0;
        for (int i = 1; i <= vec4Array.length; ++i) {
            Vec4 vec42;
            boolean bl2;
            if (bl) {
                arrayList.add(vec4);
            }
            boolean bl3 = bl2 = plane.dot(vec42 = vec4Array[i % vec4Array.length]) <= 0.0;
            if (bl != bl2) {
                Vec4[] vec4Array2 = bl ? plane.clip(vec4, vec42) : plane.clip(vec42, vec4);
                arrayList.add(vec4Array2[0]);
            }
            vec4 = vec42;
            bl = bl2;
        }
        if (arrayList.size() == 0) {
            return null;
        }
        Vec4[] vec4Array3 = new Vec4[arrayList.size()];
        return arrayList.toArray(vec4Array3);
    }

    private void addBoundaryEdges(Vec4[] vec4Array, Vec4[] vec4Array2, ArrayList<SectorGeometry.BoundaryEdge> arrayList) {
        for (int i = 0; i < vec4Array.length; ++i) {
            int n = (i + 1) % vec4Array.length;
            if (this.edgeOnTriangle(vec4Array[i], vec4Array[n], vec4Array2)) continue;
            arrayList.add(new SectorGeometry.BoundaryEdge(vec4Array, i, n));
        }
    }

    private boolean edgeOnTriangle(Vec4 vec4, Vec4 vec42, Vec4[] vec4Array) {
        double[] dArray = RectangularTessellator.baryCentricCoordsRequireInside(vec4, vec4Array);
        double[] dArray2 = RectangularTessellator.baryCentricCoordsRequireInside(vec42, vec4Array);
        if (dArray == null || dArray2 == null) {
            return true;
        }
        for (int i = 0; i < 3; ++i) {
            if (!(dArray[i] < 1.0E-4) || !(dArray2[i] < 1.0E-4)) continue;
            return true;
        }
        return false;
    }

    private SectorGeometry.ExtractedShapeDescription getIntersectingTessellationPieces(RectTile rectTile, Vec4 vec4, Vec4 vec42, Vec4 vec43, double d, double d2) {
        rectTile.ri.vertices.rewind();
        rectTile.ri.indices.rewind();
        Vec4 vec44 = rectTile.ri.referenceCenter;
        if (vec44 == null) {
            vec44 = new Vec4(0.0);
        }
        int n = rectTile.ri.indices.capacity() - 2;
        int[] nArray = new int[3];
        double[] dArray = new double[3];
        SectorGeometry.ExtractedShapeDescription extractedShapeDescription = null;
        for (int i = 0; i < n; ++i) {
            rectTile.ri.indices.position(i);
            rectTile.ri.indices.get(nArray);
            if (nArray[0] == nArray[1] || nArray[0] == nArray[2] || nArray[1] == nArray[2]) continue;
            Vec4[] vec4Array = new Vec4[3];
            for (int j = 0; j < 3; ++j) {
                rectTile.ri.vertices.position(3 * nArray[j]);
                rectTile.ri.vertices.get(dArray);
                vec4Array[j] = new Vec4(dArray[0] + vec44.getX(), dArray[1] + vec44.getY(), dArray[2] + vec44.getZ(), 1.0);
            }
            extractedShapeDescription = this.addClippedPolygon(vec4Array, vec4, vec42, vec43, d, d2, extractedShapeDescription);
        }
        return extractedShapeDescription;
    }

    private SectorGeometry.ExtractedShapeDescription addClippedPolygon(Vec4[] vec4Array, Vec4 vec4, Vec4 vec42, Vec4 vec43, double d, double d2, SectorGeometry.ExtractedShapeDescription extractedShapeDescription) {
        if (this.isSkirt(vec4Array)) {
            return extractedShapeDescription;
        }
        int n = 0;
        int n2 = 0;
        int n3 = -1;
        int n4 = -1;
        for (Vec4 vec44 : vec4Array) {
            double d3;
            Vec4 vec45 = vec44.subtract3(vec4);
            double d4 = vec45.dot3(vec42);
            double d5 = d4 * d4 / (d * d) + (d3 = vec45.dot3(vec43)) * d3 / (d2 * d2) - 1.0;
            if (d5 <= 0.0) {
                n3 = n++;
                ++n2;
                continue;
            }
            n4 = n++;
        }
        SectorGeometry.BoundaryEdge boundaryEdge = new SectorGeometry.BoundaryEdge(null, -1, -1);
        switch (n2) {
            case 0: {
                vec4Array = this.checkForEdgeCylinderIntersections(vec4Array, vec4, vec42, vec43, d, d2);
                break;
            }
            case 1: {
                if (n3 != 0) {
                    Vec4 vec47 = vec4Array[n3];
                    vec4Array[n3] = vec4Array[0];
                    vec4Array[0] = vec47;
                }
                vec4Array = this.computeTrimmedPoly(vec4Array, vec4, vec42, vec43, d, d2, n2, boundaryEdge);
                break;
            }
            case 2: {
                if (n4 != 0) {
                    Vec4 vec46 = vec4Array[n4];
                    vec4Array[n4] = vec4Array[0];
                    vec4Array[0] = vec46;
                }
                vec4Array = this.computeTrimmedPoly(vec4Array, vec4, vec42, vec43, d, d2, n2, boundaryEdge);
                break;
            }
        }
        if (vec4Array == null) {
            return extractedShapeDescription;
        }
        if (extractedShapeDescription == null) {
            extractedShapeDescription = new SectorGeometry.ExtractedShapeDescription(new ArrayList<Vec4[]>(100), new ArrayList<SectorGeometry.BoundaryEdge>(50));
        }
        extractedShapeDescription.interiorPolys.add(vec4Array);
        if (boundaryEdge.vertices != null) {
            extractedShapeDescription.shapeOutline.add(boundaryEdge);
        }
        return extractedShapeDescription;
    }

    private Vec4[] checkForEdgeCylinderIntersections(Vec4[] vec4Array, Vec4 vec4, Vec4 vec42, Vec4 vec43, double d, double d2) {
        return null;
    }

    private Vec4[] computeTrimmedPoly(Vec4[] vec4Array, Vec4 vec4, Vec4 vec42, Vec4 vec43, double d, double d2, int n, SectorGeometry.BoundaryEdge boundaryEdge) {
        Vec4 vec44 = this.intersectWithEllCyl(vec4Array[0], vec4Array[1], vec4, vec42, vec43, d, d2);
        Vec4 vec45 = this.intersectWithEllCyl(vec4Array[0], vec4Array[2], vec4, vec42, vec43, d, d2);
        Vec4 vec46 = vec44.multiply3(0.5).add3(vec45.multiply3(0.5));
        if (n == 1) {
            vec4Array[1] = vec44;
            vec4Array[2] = vec45;
            boundaryEdge.vertices = vec4Array;
            boundaryEdge.i1 = 1;
            boundaryEdge.i2 = 2;
            boundaryEdge.toMidPoint = vec46.subtract3(vec4Array[0]);
            return vec4Array;
        }
        Vec4[] vec4Array2 = new Vec4[]{vec44, vec4Array[1], vec4Array[2], vec45};
        boundaryEdge.vertices = vec4Array2;
        boundaryEdge.i1 = 0;
        boundaryEdge.i2 = 3;
        boundaryEdge.toMidPoint = vec4Array[0].subtract3(vec46);
        return vec4Array2;
    }

    private Vec4 intersectWithEllCyl(Vec4 vec4, Vec4 vec42, Vec4 vec43, Vec4 vec44, Vec4 vec45, double d, double d2) {
        double d3;
        double d4;
        double d5;
        double d6;
        double d7;
        double d8;
        double d9;
        Vec4 vec46 = vec4.subtract3(vec43);
        double d10 = vec46.dot3(vec44);
        double d11 = vec46.dot3(vec45);
        Vec4 vec47 = vec42.subtract3(vec43);
        double d12 = vec47.dot3(vec44);
        double d13 = d12 - d10;
        double d14 = 2.0 * (d10 * d13 / (d9 = d * d) + d11 * (d8 = (d7 = vec47.dot3(vec45)) - d11) / (d6 = d2 * d2));
        double d15 = (-d14 + (d5 = Math.sqrt(d14 * d14 - 4.0 * (d4 = d13 * d13 / d9 + d8 * d8 / d6) * (d3 = d10 * d10 / d9 + d11 * d11 / d6 - 1.0)))) / (2.0 * d4);
        if (d15 < 0.0 || d15 > 1.0) {
            d15 = (-d14 - d5) / (2.0 * d4);
        }
        return vec4.multiply3(1.0 - d15).add3(vec42.multiply3(d15));
    }

    public static RectGeometry getTerrainGeometry(SectorGeometry sectorGeometry) {
        if (sectorGeometry == null || !(sectorGeometry instanceof RectTile)) {
            throw new IllegalArgumentException("SectorGeometry instance not of type RectTile");
        }
        return new RectGeometry((RectTile)sectorGeometry);
    }

    public static class RectGeometry {
        private RectTile tile;
        private double rowFactor;
        private double colFactor;

        public RectGeometry(RectTile rectTile) {
            this.tile = rectTile;
            this.rowFactor = (double)this.getNumRows() / rectTile.sector.getDeltaLatDegrees();
            this.colFactor = (double)this.getNumCols() / rectTile.sector.getDeltaLonDegrees();
        }

        public int getColAtLon(double d) {
            return (int)Math.floor((d - ((RectTile)this.tile).sector.getMinLongitude().degrees) * this.colFactor);
        }

        public int getRowAtLat(double d) {
            return (int)Math.floor((d - ((RectTile)this.tile).sector.getMinLatitude().degrees) * this.rowFactor);
        }

        public double getLatAtRow(int n) {
            return ((RectTile)this.tile).sector.getMinLatitude().degrees + (double)n / this.rowFactor;
        }

        public double getLonAtCol(int n) {
            return ((RectTile)this.tile).sector.getMinLongitude().degrees + (double)n / this.colFactor;
        }

        public double[] getPointAt(double d, double d2) {
            int n = this.getColAtLon(d2);
            if (n < 0) {
                n = 0;
                d2 = this.getMinLongitude();
            } else if (n > this.getNumCols()) {
                n = this.getNumCols();
                d2 = this.getMaxLongitude();
            }
            int n2 = this.getRowAtLat(d);
            if (n2 < 0) {
                n2 = 0;
                d = this.getMinLatitude();
            } else if (n2 > this.getNumRows()) {
                n2 = this.getNumRows();
                d = this.getMaxLatitude();
            }
            double[] dArray = new double[3];
            this.tile.ri.vertices.position(this.getVertexIndex(n2, n));
            this.tile.ri.vertices.get(dArray);
            double[] dArray2 = new double[3];
            this.tile.ri.vertices.position(this.getVertexIndex(n2, n + 1));
            this.tile.ri.vertices.get(dArray2);
            double[] dArray3 = new double[3];
            this.tile.ri.vertices.position(this.getVertexIndex(n2 + 1, n));
            this.tile.ri.vertices.get(dArray3);
            double[] dArray4 = new double[3];
            this.tile.ri.vertices.position(this.getVertexIndex(n2 + 1, n + 1));
            this.tile.ri.vertices.get(dArray4);
            double[] dArray5 = new double[3];
            this.tile.ri.referenceCenter.toArray3(dArray5, 0);
            double d3 = this.getLonAtCol(n);
            double d4 = this.getLonAtCol(n + 1);
            double d5 = this.getLatAtRow(n2);
            double d6 = this.getLatAtRow(n2 + 1);
            double d7 = (d2 - d3) / (d4 - d3);
            double d8 = (d - d5) / (d6 - d5);
            double[] dArray6 = new double[]{dArray[0] * (1.0 - d7) * (1.0 - d8) + dArray2[0] * d7 * (1.0 - d8) + dArray3[0] * (1.0 - d7) * d8 + dArray4[0] * d7 * d8 + dArray5[0], dArray[1] * (1.0 - d7) * (1.0 - d8) + dArray2[1] * d7 * (1.0 - d8) + dArray3[1] * (1.0 - d7) * d8 + dArray4[1] * d7 * d8 + dArray5[1], dArray[2] * (1.0 - d7) * (1.0 - d8) + dArray2[2] * d7 * (1.0 - d8) + dArray3[2] * (1.0 - d7) * d8 + dArray4[2] * d7 * d8 + dArray5[2]};
            return dArray6;
        }

        public double getMinLongitude() {
            return ((RectTile)this.tile).sector.getMinLongitude().degrees;
        }

        public double getMaxLongitude() {
            return ((RectTile)this.tile).sector.getMaxLongitude().degrees;
        }

        public double getMinLatitude() {
            return ((RectTile)this.tile).sector.getMinLatitude().degrees;
        }

        public double getMaxLatitude() {
            return ((RectTile)this.tile).sector.getMaxLatitude().degrees;
        }

        public int getNumRows() {
            return this.tile.density;
        }

        public int getNumCols() {
            return this.tile.density;
        }

        private int getVertexIndex(int n, int n2) {
            return (this.tile.density + 3) * (n + 1) * 3 + (n2 + 1) * 3;
        }
    }

    private static class CacheKey {
        private final Sector sector;
        private final int density;
        private final Object globeStateKey;

        public CacheKey(DrawContext drawContext, Sector sector, int n) {
            this.sector = sector;
            this.density = n;
            this.globeStateKey = drawContext.getGlobe().getStateKey(drawContext);
        }

        public boolean equals(Object object) {
            if (this == object) {
                return true;
            }
            CacheKey cacheKey = (CacheKey)object;
            if (this.density != cacheKey.density) {
                return false;
            }
            if (this.globeStateKey != null ? !this.globeStateKey.equals(cacheKey.globeStateKey) : cacheKey.globeStateKey != null) {
                return false;
            }
            return !(this.sector != null ? !this.sector.equals(cacheKey.sector) : cacheKey.sector != null);
        }

        public int hashCode() {
            int n = this.sector != null ? this.sector.hashCode() : 0;
            n = 31 * n + this.density;
            n = 31 * n + (this.globeStateKey != null ? this.globeStateKey.hashCode() : 0);
            return n;
        }
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    private static class RectTile
    implements SectorGeometry {
        private final RectangularTessellator tessellator;
        private final int level;
        private final Sector sector;
        private final int density;
        private final double log10CellSize;
        private Extent extent;
        private RenderInfo ri;
        private int minColorCode = 0;
        private int maxColorCode = 0;

        public RectTile(RectangularTessellator rectangularTessellator, Extent extent, int n, int n2, Sector sector, double d) {
            this.tessellator = rectangularTessellator;
            this.level = n;
            this.density = n2;
            this.sector = sector;
            this.extent = extent;
            this.log10CellSize = Math.log10(d);
        }

        @Override
        public Sector getSector() {
            return this.sector;
        }

        @Override
        public Extent getExtent() {
            return this.extent;
        }

        @Override
        public void renderMultiTexture(DrawContext drawContext, int n) {
            this.tessellator.renderMultiTexture(drawContext, this, n);
        }

        @Override
        public void render(DrawContext drawContext) {
            this.tessellator.render(drawContext, this);
        }

        @Override
        public void renderWireframe(DrawContext drawContext, boolean bl, boolean bl2) {
            this.tessellator.renderWireframe(drawContext, this, bl, bl2);
        }

        @Override
        public void renderBoundingVolume(DrawContext drawContext) {
            this.tessellator.renderBoundingVolume(drawContext, this);
        }

        @Override
        public PickedObject[] pick(DrawContext drawContext, List<? extends Point> list) {
            return this.tessellator.pick(drawContext, this, list);
        }

        @Override
        public void pick(DrawContext drawContext, Point point) {
            this.tessellator.pick(drawContext, this, point);
        }

        @Override
        public Vec4 getSurfacePoint(Angle angle, Angle angle2, double d) {
            return this.tessellator.getSurfacePoint(this, angle, angle2, d);
        }

        public double getResolution() {
            return this.sector.getDeltaLatRadians() / (double)this.density;
        }

        @Override
        public Intersection[] intersect(Line line) {
            return this.tessellator.intersect(this, line);
        }

        @Override
        public Intersection[] intersect(double d) {
            return this.tessellator.intersect(this, d);
        }

        @Override
        public DoubleBuffer makeTextureCoordinates(SectorGeometry.GeographicTextureCoordinateComputer geographicTextureCoordinateComputer) {
            return this.tessellator.makeGeographicTexCoords(this, geographicTextureCoordinateComputer);
        }

        @Override
        public SectorGeometry.ExtractedShapeDescription getIntersectingTessellationPieces(Plane[] planeArray) {
            return this.tessellator.getIntersectingTessellationPieces(this, planeArray);
        }

        @Override
        public SectorGeometry.ExtractedShapeDescription getIntersectingTessellationPieces(Vec4 vec4, Vec4 vec42, Vec4 vec43, double d, double d2) {
            return this.tessellator.getIntersectingTessellationPieces(this, vec4, vec42, vec43, d, d2);
        }
    }

    protected static class RenderInfo {
        private final int density;
        private final Vec4 referenceCenter;
        private final DoubleBuffer vertices;
        private final DoubleBuffer texCoords;
        private final IntBuffer indices;
        private final long time;

        private RenderInfo(int n, DoubleBuffer doubleBuffer, DoubleBuffer doubleBuffer2, Vec4 vec4) {
            this.density = n;
            this.vertices = doubleBuffer;
            this.texCoords = doubleBuffer2;
            this.referenceCenter = vec4;
            this.indices = RectangularTessellator.getIndices(this.density);
            this.time = System.currentTimeMillis();
        }

        private long getSizeInBytes() {
            return 32 + this.vertices.limit() * 64;
        }
    }
}

